"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Service = void 0;
/* eslint-disable @typescript-eslint/ban-ts-comment */
const errors_1 = require("@feathersjs/errors");
const commons_1 = require("@feathersjs/commons");
const debug = (0, commons_1.createDebug)('@feathersjs/transport-commons/client');
const namespacedEmitterMethods = [
    'addListener',
    'addEventListener',
    'emit',
    'listenerCount',
    'listeners',
    'on',
    'once',
    'prependListener',
    'prependOnceListener',
    'removeAllListeners',
    'removeEventListener',
    'removeListener'
];
const otherEmitterMethods = ['eventNames', 'getMaxListeners', 'setMaxListeners'];
const addEmitterMethods = (service) => {
    otherEmitterMethods.forEach((method) => {
        service[method] = function (...args) {
            if (typeof this.connection[method] !== 'function') {
                throw new Error(`Can not call '${method}' on the client service connection`);
            }
            return this.connection[method](...args);
        };
    });
    // Methods that should add the namespace (service path)
    namespacedEmitterMethods.forEach((method) => {
        service[method] = function (name, ...args) {
            if (typeof this.connection[method] !== 'function') {
                throw new Error(`Can not call '${method}' on the client service connection`);
            }
            const eventName = `${this.path} ${name}`;
            debug(`Calling emitter method ${method} with ` + `namespaced event '${eventName}'`);
            const result = this.connection[method](eventName, ...args);
            return result === this.connection ? this : result;
        };
    });
};
class Service {
    constructor(options) {
        this.events = options.events;
        this.path = options.name;
        this.connection = options.connection;
        this.method = options.method;
        addEmitterMethods(this);
    }
    send(method, ...args) {
        return new Promise((resolve, reject) => {
            var _a, _b;
            const route = args.pop();
            let path = this.path;
            if (route) {
                Object.keys(route).forEach((key) => {
                    path = path.replace(`:${key}`, route[key]);
                });
            }
            args.unshift(method, path);
            const socketTimeout = ((_a = this.connection.flags) === null || _a === void 0 ? void 0 : _a.timeout) || ((_b = this.connection._opts) === null || _b === void 0 ? void 0 : _b.ackTimeout);
            if (socketTimeout !== undefined) {
                args.push(function (timeoutError, error, data) {
                    return timeoutError || error ? reject((0, errors_1.convert)(timeoutError || error)) : resolve(data);
                });
            }
            else {
                args.push(function (error, data) {
                    return error ? reject((0, errors_1.convert)(error)) : resolve(data);
                });
            }
            debug(`Sending socket.${this.method}`, args);
            this.connection[this.method](...args);
        });
    }
    methods(...names) {
        names.forEach((method) => {
            const _method = `_${method}`;
            this[_method] = function (data, params = {}) {
                return this.send(method, data, params.query || {}, params.route || {});
            };
            this[method] = function (data, params = {}) {
                return this[_method](data, params, params.route || {});
            };
        });
        return this;
    }
    _find(params = {}) {
        return this.send('find', params.query || {}, params.route || {});
    }
    find(params = {}) {
        return this._find(params);
    }
    _get(id, params = {}) {
        return this.send('get', id, params.query || {}, params.route || {});
    }
    get(id, params = {}) {
        return this._get(id, params);
    }
    _create(data, params = {}) {
        return this.send('create', data, params.query || {}, params.route || {});
    }
    create(data, params = {}) {
        return this._create(data, params);
    }
    _update(id, data, params = {}) {
        if (typeof id === 'undefined') {
            return Promise.reject(new Error("id for 'update' can not be undefined"));
        }
        return this.send('update', id, data, params.query || {}, params.route || {});
    }
    update(id, data, params = {}) {
        return this._update(id, data, params);
    }
    _patch(id, data, params = {}) {
        return this.send('patch', id, data, params.query || {}, params.route || {});
    }
    patch(id, data, params = {}) {
        return this._patch(id, data, params);
    }
    _remove(id, params = {}) {
        return this.send('remove', id, params.query || {}, params.route || {});
    }
    remove(id, params = {}) {
        return this._remove(id, params);
    }
    // `off` is actually not part of the Node event emitter spec
    // but we are adding it since everybody is expecting it because
    // of the emitter-component Socket.io is using
    off(name, ...args) {
        if (typeof this.connection.off === 'function') {
            const result = this.connection.off(`${this.path} ${name}`, ...args);
            return result === this.connection ? this : result;
        }
        else if (args.length === 0) {
            // @ts-ignore
            return this.removeAllListeners(name);
        }
        // @ts-ignore
        return this.removeListener(name, ...args);
    }
}
exports.Service = Service;
//# sourceMappingURL=client.js.map